﻿# -*- coding: utf-8 -*-

import bpy
from bgl import glEnable, GL_BLEND
import blf
from gpu.types import GPUShader, GPUBatch, GPUVertBuf, GPUIndexBuf, GPUVertFormat

#### =======================================================================
class View3dBorderDeco(object):
	'''View3dを額縁修飾する'''
	__slots__ = (
		"hdl", "vbo", "ibo", "vtFmt", "vtPos", "shader",
		"colors", "width", "offset", "showLabel", "ignoreCamera",
		"isDirtyColor", "isDirtyShape", "rx", "ry", "rmode",
		)

	#### ----------------
	vtIdx = ((0,1,5),(0,5,4), (1,2,6),(1,6,5), (2,3,7),(2,7,6), (3,0,4),(3,4,7))
	vs = ("uniform mat4 ModelViewProjectionMatrix;in vec2 pos;in vec4 col;out vec4 outColr;"
		  "void main(){gl_Position=ModelViewProjectionMatrix*vec4(pos,0f,1f);outColr=col;}")
	fs = ("in vec4 outColr;void main(){gl_FragColor=outColr;}")

	#### ----------------
	class COLR(object):
		OBJ  = ((1.0, 1.0, 1.0, 0.4),)*8
		MESH = ((0.3, 0.7, 0.3, 0.4),)*8
		CRV  = MESH
		SURF = MESH
		BONE = ((1.0,  0.15, 0.15, 0.4),)*8
		TEXT = ((0.5,  0.4,  1.0,  0.4),)*8
		BALL = ((0.17, 0.75, 0.75, 0.4),)*8
		LATT = ((0.2,  0.0,  0.15, 0.4),)*8
		GPEN = ((0.25, 0.45, 0.65, 0.4),)*8
		POSE = ((0.9,  0.9,  0.2,  0.4),)*8
		SCLP = ((0.0,  0.15, 0.6,  0.4),)*8
		PWGT = ((0.3,  0.4,  0.6,  0.4),)*8
		PVTX = PWGT
		PTEX = PWGT
		PGP  = ((0.5,  0.65, 0.7,  0.4),)*8
		WGP  = PWGT
		SGP  = SCLP
		PRTL = ((0.0,  1.0,  0.0,  0.4),)*8
		__slots__=()

	#### ----------------
	initialStatsDct = {
		"OBJECT": (False, COLR.OBJ),
		"EDIT_MESH": (False, COLR.MESH),
		"EDIT_CURVE": (False, COLR.CRV),
		"EDIT_SURFACE": (False, COLR.SURF),
		"EDIT_ARMATURE": (False, COLR.BONE),
		"EDIT_TEXT": (False, COLR.TEXT),
		"EDIT_METABALL": (False, COLR.BALL),
		"EDIT_LATTICE": (False, COLR.LATT),
		"EDIT_GPENCIL": (False, COLR.GPEN),
		"POSE": (False, COLR.POSE),
		"SCULPT": (False, COLR.SCLP),
		"PAINT_WEIGHT": (False, COLR.PWGT),
		"PAINT_VERTEX": (False, COLR.PVTX),
		"PAINT_TEXTURE": (False, COLR.PTEX),
		"PAINT_GPENCIL": (False, COLR.PGP),
		"WEIGHT_GPENCIL": (False, COLR.WGP),
		"SCULPT_GPENCIL": (False, COLR.SGP),
		"PARTICLE": (False, COLR.PRTL),
		}

	#### --------------------------------
	def __init__(self):
		self.colors = self.initialStatsDct.copy()	# key:tuple, 書き換えはtuple単位
		self.MakeShaders()

		self.width = 4
		self.offset = 0
		self.showLabel = True
		self.ignoreCamera = True

		self.isDirtyColor = self.isDirtyShape = True
		self.rmode = ""
		self.rx = self.ry = 0

		self.hdl = bpy.types.SpaceView3D.draw_handler_add(self.draw, (), "WINDOW", "POST_PIXEL")

	#### --------------------------------
	def Update_Flags(self, **kwd):
		'''外部からの更新用'''
		v = kwd.get("borderWidth", kwd.get("bw", None))
		if(v != None):
			self.width = max(min(v, 100), 1)
			self.isDirtyShape = True

		v = kwd.get("borderOffset", kwd.get("bo", None))
		if(v != None):
			self.offset = max(min(v, 100), 0)
			self.isDirtyShape = True

		v = kwd.get("showLabel", kwd.get("sl", None))
		if(v != None):
			self.showLabel = v

		v = kwd.get("ignoreCamera", kwd.get("ic", None))
		if(v != None):
			self.ignoreCamera = v

	#### --------------------------------
	def Update_Stat(self, tag, stat, colr):
		'''外部からの更新用'''
		self.colors[tag] = (stat, ((colr,) * 8))
		self.isDirtyColor = True

	#### --------------------------------
	def MakeShaders(self):
		'''全て一度の初期化で再使用する.'''
		self.shader = GPUShader(self.vs, self.fs)
		self.vtFmt = GPUVertFormat()
		self.vtFmt.attr_add(id="pos", comp_type="F32", len=2, fetch_mode="FLOAT")
		self.vtFmt.attr_add(id="col", comp_type="F32", len=4, fetch_mode="FLOAT")
		self.ibo = GPUIndexBuf(type= "TRIS", seq= self.vtIdx)

	#### --------------------------------
	def MakeVBO(self, colr):
		'''attr_fillで既設要素を上書きは出来ない, vboの再設定不可. '''
		self.vbo = GPUVertBuf(self.vtFmt, 8)	# len(vtPos)
		self.vbo.attr_fill("pos", self.vtPos)
		self.vbo.attr_fill("col", colr)

	#### --------------------------------
	def MakeVtPos(self, space, pref):
		'''頂点配列作成. xyzwの2要素分のみ. ((x,y,-,-),(,,,),)'''
		xoMin = self.offset
		yoMin = self.offset
		xoMax = self.rx - self.offset
		yoMax = self.ry - self.offset
		xiMin = self.width + self.offset
		yiMin = self.width + self.offset
		xiMax = xoMax - self.width
		yiMax = yoMax - self.width - 1

		if(pref.system.use_region_overlap):
			xiMin += 1
			yiMin += 1
			xiMax -= 1
			fScl = pref.view.ui_scale
			if(space.show_region_ui):		# sideBarタブ幅
				v = 20 * fScl
				xoMax -= v
				xiMax -= v
			if(space.show_region_header):	# ヘッダ厚context.region.width-20
				v = 25 * fScl
				yoMin += v
				yiMin += v
		else:
			xoMax += 1
			if(not space.show_region_ui):
				xiMax -= 1

		self.vtPos = ((xoMin, yoMin), (xoMin, yoMax), (xoMax, yoMax), (xoMax, yoMin),
				(xiMin, yiMin), (xiMin, yiMax), (xiMax, yiMax), (xiMax, yiMin),)

	#### --------------------------------
	def Font_Draw(self, context):
		mode = context.mode
		fScl = context.preferences.view.ui_scale
		id = 0

		x = context.region.width
		if(context.preferences.system.use_region_overlap and context.space_data.show_region_ui):
			x -= (20 * fScl)
		blf.size(id, int(11 * fScl), 72)
		blf.shadow_offset(id, 1, -1)
		blf.enable(id, blf.SHADOW)
		(dx, _) = blf.dimensions(id, mode)
		blf.position(id, (x-dx)/2, context.region.height - (20 * fScl), 0)
		blf.draw(id, mode)
		blf.disable(id, blf.SHADOW)

	#### --------------------------------
	def draw(self):
		context = bpy.context
		if((context.region_data.view_perspective == "CAMERA") and self.ignoreCamera):
			pass
		else:
			mode = context.mode
			if(self.colors[mode][0]):
				if((self.rx != context.region.width) or (self.ry != context.region.height) or self.isDirtyShape):
					self.rx = context.region.width
					self.ry = context.region.height
					self.MakeVtPos(context.space_data, context.preferences)
					self.isDirtyShape = True

				if((self.rmode != mode) or self.isDirtyColor):
					self.rmode = mode
					self.isDirtyColor = True

				if(self.isDirtyColor or self.isDirtyShape):
					self.MakeVBO(self.colors[mode][1])
					self.isDirtyShape = self.isDirtyColor = False

				glEnable(GL_BLEND)
				GPUBatch(type= "TRIS", buf= self.vbo, elem= self.ibo).draw(self.shader)

		if(self.showLabel):
			self.Font_Draw(context)

	#### --------------------------------
	def unregist(self):
		bpy.types.SpaceView3D.draw_handler_remove(self.hdl, "WINDOW")

